#include "winxy.h"
#include "./ui_winxy.h"

// Include your modules here
#include "scheduled_shutdown.h"
#include "optimize_performance.h"
#include "context_menu.h"

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QScrollArea>
#include <QLabel>
#include <QFile>
#include <QTextStream>
#include <QDateTime>
#include <QDir>
#include <QMutex>
#include <QCoreApplication>
#include <QDebug>
#include <QFileInfo>

// =============================================================================
// [NEW] AppConfig Implementation (Dynamic Resource Discovery)
// =============================================================================

QString AppConfig::m_rootDir;

void AppConfig::initialize()
{
    // Anchor: The executable's current directory
    QString appPath = QCoreApplication::applicationDirPath();
    QDir appDir(appPath);

    // Search Strategy: Look for the "figures" folder in:
    // 1. Current Dir (.)
    // 2. Parent Dir (..)
    // 3. Grandparent Dir (../..)
    QStringList searchLevels = { ".", "..", "../.." };
    bool found = false;

    for (const QString &level : searchLevels) {
        QString candidate = appPath + "/" + level + "/figures";
        QDir checkDir(candidate);

        if (checkDir.exists()) {
            // Found it! The project root is the parent of "figures"
            m_rootDir = QFileInfo(candidate).absolutePath(); // This gives the folder CONTAINING figures
            // Adjust: if candidate is ".../winxy/figures", absolutePath is ".../winxy"
            // Wait, QFileInfo("/path/to/figures").absolutePath() returns "/path/to".
            // That is exactly what we want (the project root).
            m_rootDir = QDir::cleanPath(m_rootDir);
            found = true;
            qDebug() << "[AppConfig] Project root discovered at:" << m_rootDir;
            break;
        }
    }

    if (!found) {
        // Fallback: Just use the executable directory
        m_rootDir = appPath;
        qWarning() << "[AppConfig] 'figures' folder not found. Defaulting root to:" << m_rootDir;
    }
}

QString AppConfig::getRootDir()
{
    return m_rootDir;
}

QString AppConfig::getAssetPath(const QString &fileName)
{
    // Construct path: Root/figures/fileName
    return QDir(m_rootDir).filePath("figures/" + fileName);
}

QString AppConfig::getRuleJsonPath()
{
    // Construct path: Root/rule.json
    return QDir(m_rootDir).filePath("rule.json");
}

// =============================================================================
// Logging Logic
// =============================================================================

static QFile *g_logFile = nullptr;

void customMessageHandler(QtMsgType type, const QMessageLogContext &context, const QString &msg)
{
    if (!g_logFile || !g_logFile->isOpen()) return;

    static QMutex mutex;
    QMutexLocker locker(&mutex);

    QTextStream stream(g_logFile);
    QString timestamp = QDateTime::currentDateTime().toString("yyyy-MM-dd HH:mm:ss");

    QString typeStr;
    switch (type) {
    case QtDebugMsg:    typeStr = "[DEBUG]"; break;
    case QtInfoMsg:     typeStr = "[INFO] "; break;
    case QtWarningMsg:  typeStr = "[WARN] "; break;
    case QtCriticalMsg: typeStr = "[CRIT] "; break;
    case QtFatalMsg:    typeStr = "[FATAL]"; break;
    }
    stream << QString("[%1] %2 %3").arg(timestamp, typeStr, msg) << Qt::endl;
}

void initLogger()
{
    // Log file always stays with the executable for easy access
    QString logPath = QCoreApplication::applicationDirPath() + "/debug.log";
    g_logFile = new QFile(logPath);

    if (g_logFile->open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate)) {
        qInstallMessageHandler(customMessageHandler);
        qDebug() << "Logger initialized. Log file path:" << logPath;
    } else {
        qWarning() << "Failed to open log file at:" << logPath;
    }
}

// =============================================================================
// WinXY Implementation
// =============================================================================

WinXY::WinXY(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::WinXY)
{
    // Initialize Logger
    initLogger();

    ui->setupUi(this);

    this->setWindowTitle(QString("WinXY Toolkit %1").arg(qApp->applicationVersion()));
    this->resize(900, 600);

    QWidget *centralWidget = new QWidget(this);
    this->setCentralWidget(centralWidget);
    QHBoxLayout *mainLayout = new QHBoxLayout(centralWidget);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);

    // Sidebar
    sidebarContainer = new QWidget(this);
    sidebarContainer->setFixedWidth(250);
    sidebarContainer->setStyleSheet("background-color: #f3f3f3; border-right: 1px solid #e0e0e0;");
    sidebarLayout = new QVBoxLayout(sidebarContainer);
    sidebarLayout->setContentsMargins(0, 10, 0, 0);
    sidebarLayout->setSpacing(0);
    mainLayout->addWidget(sidebarContainer);

    // Content
    QWidget *contentContainer = new QWidget(this);
    contentContainer->setStyleSheet("background-color: #ffffff;");
    QVBoxLayout *contentLayout = new QVBoxLayout(contentContainer);
    contentLayout->setContentsMargins(0,0,0,0);

    mainContentStack = new QStackedWidget(this);
    contentLayout->addWidget(mainContentStack);
    mainLayout->addWidget(contentContainer);

    // Register Modules
    addFeature(new ScheduledShutdownWidget(this), "Scheduled Shutdown");
    addFeature(new OptimizePerformanceWidget(this), "System Optimization");
    addFeature(new ContextMenuWidget(this), "Context Menu Manager");

    sidebarLayout->addStretch();

    // [UPDATED] Version Info & Feedback Display
    QLabel *versionInfo = new QLabel(this);
    versionInfo->setTextFormat(Qt::RichText);
    versionInfo->setOpenExternalLinks(true); // Allows clicking mailto: links if you ever add them

    // Retrieve the custom "FeedbackEmail" property set in main.cpp
    QString feedback = qApp->property("FeedbackEmail").toString();

    versionInfo->setText(QString("Ver %1<br>© 2026 %2<br><br>Feedback:<br>%3")
                             .arg(qApp->applicationVersion(),
                                  qApp->organizationName(),
                                  feedback));

    versionInfo->setStyleSheet("QLabel { color: #888888; font-size: 11px; padding-left: 25px; padding-bottom: 15px; }");
    versionInfo->setAlignment(Qt::AlignBottom | Qt::AlignLeft);
    sidebarLayout->addWidget(versionInfo);

    if (!sidebarTabs.isEmpty()) {
        sidebarTabs[0]->click();
    }
}

WinXY::~WinXY()
{
    delete ui;
    if (g_logFile) {
        g_logFile->close();
        delete g_logFile;
    }
}

void WinXY::addFeature(QWidget *pageWidget, const QString &tabName)
{
    QScrollArea *scrollArea = new QScrollArea(this);
    scrollArea->setWidgetResizable(true);
    scrollArea->setFrameShape(QFrame::NoFrame);
    scrollArea->setStyleSheet("QScrollArea { background-color: transparent; border: none; }");
    scrollArea->setWidget(pageWidget);

    mainContentStack->addWidget(scrollArea);
    int index = mainContentStack->count() - 1;

    QPushButton *btn = createSidebarButton(tabName);
    btn->setProperty("pageIndex", index);
    connect(btn, &QPushButton::clicked, this, &WinXY::onTabClicked);

    sidebarLayout->addWidget(btn);
    sidebarTabs.append(btn);
}

QPushButton* WinXY::createSidebarButton(const QString &text)
{
    QPushButton *btn = new QPushButton(text, this);
    btn->setFixedHeight(45);
    btn->setCursor(Qt::PointingHandCursor);
    return btn;
}

void WinXY::onTabClicked()
{
    QPushButton *senderBtn = qobject_cast<QPushButton*>(sender());
    if (!senderBtn) return;

    int index = senderBtn->property("pageIndex").toInt();
    mainContentStack->setCurrentIndex(index);
    updateTabStyles(index);
}

void WinXY::updateTabStyles(int activeIndex)
{
    QString baseStyle =
        "QPushButton { border: none; text-align: left; padding-left: 15px; color: #555; background-color: transparent; }"
        "QPushButton:hover { background-color: #e8e8e8; }";

    QString activeStyle =
        "QPushButton { border: none; border-left: 4px solid #0078d4; text-align: left; padding-left: 11px; color: #333; font-weight: bold; background-color: #e8e8e8; }";

    for (int i = 0; i < sidebarTabs.size(); ++i) {
        if (i == activeIndex) {
            sidebarTabs[i]->setStyleSheet(activeStyle);
        } else {
            sidebarTabs[i]->setStyleSheet(baseStyle);
        }
    }
}
